package mod.common.block.entity;



import mod.common.recipes.AnalyzerRecipes;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.inventory.IInventory;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.nbt.NBTTagList;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.IChatComponent;
import net.minecraftforge.fml.relauncher.Side;
import net.minecraftforge.fml.relauncher.SideOnly;

public class TileEntityAnalyzer extends TileEntity implements IInventory
{
    private ItemStack[] contents = new ItemStack[2];

    private int workingTime = 1999;
    private int workingTimeNeeded = 256;

    @Override
    public void writeToNBT(NBTTagCompound compound) {
        super.writeToNBT(compound);
        NBTTagList nbttaglist = new NBTTagList();

        for (int i = 0; i < this.contents.length; ++i) {
            if (this.contents[i] != null) {
                NBTTagCompound nbttagcompound1 = new NBTTagCompound();
                nbttagcompound1.setByte("Slot", (byte) i);
                this.contents[i].writeToNBT(nbttagcompound1);
                nbttaglist.appendTag(nbttagcompound1);
            }
        }

        compound.setTag("Items", nbttaglist);
        compound.setShort("workingTime", (short) this.workingTime);

        compound.setShort("workingTimeNeeded", (short) this.workingTimeNeeded);
    }

    @Override
    public void readFromNBT(NBTTagCompound compound) {
        super.readFromNBT(compound);

        NBTTagList nbttaglist = compound.getTagList("Items", 10);
        this.contents = new ItemStack[this.getSizeInventory()];

        for (int i = 0; i < nbttaglist.tagCount(); ++i)

        {
            NBTTagCompound nbttagcompound1 = nbttaglist.getCompoundTagAt(i);
            int j = nbttagcompound1.getByte("Slot") & 255;

            if (j >= 0 && j < this.contents.length) {
                this.contents[j] = ItemStack.loadItemStackFromNBT(nbttagcompound1);
            }
        }

        this.workingTime = compound.getShort("workingTime");

        this.workingTimeNeeded = compound.getShort("workingTimeNeeded");
    }

    @Override
    public int getSizeInventory() {
        return this.contents.length;
    }

    @Override
    public ItemStack getStackInSlot(int slotIndex) {
        return this.contents[slotIndex];
    }

    @Override
    public ItemStack decrStackSize(int slotIndex, int amount) {
        if (this.contents[slotIndex] != null) {
            ItemStack itemstack;

            if (this.contents[slotIndex].stackSize <= amount) {
                itemstack = this.contents[slotIndex];
                this.contents[slotIndex] = null;
                this.markDirty();
                return itemstack;
            } else {
                itemstack = this.contents[slotIndex].splitStack(amount);

                if (this.contents[slotIndex].stackSize == 0) {
                    this.contents[slotIndex] = null;
                }

                this.markDirty();
                return itemstack;
            }
        } else {
            return null;
        }
    }

    @Override
    public ItemStack getStackInSlotOnClosing(int slotIndex) {
        if (this.contents[slotIndex] != null) {
            ItemStack itemstack = this.contents[slotIndex];
            this.contents[slotIndex] = null;
            return itemstack;
        } else {
            return null;
        }
    }

    @Override
    public void setInventorySlotContents(int slotIndex, ItemStack stack) {
        this.contents[slotIndex] = stack;

        if (stack != null && stack.stackSize > this.getInventoryStackLimit()) {
            stack.stackSize = this.getInventoryStackLimit();
        }

        this.markDirty();
    }


    @Override
    public boolean isItemValidForSlot(int slot, ItemStack stack) {
        return slot == 1 ? false : true;
    }

    public boolean isBurning() {
        return this.workingTime > 0;
    }

    private boolean canSmelt() {
        if (this.contents[0] == null) 
        {
            return false;
        } 
        else 
        {
            ItemStack itemstack = AnalyzerRecipes.getOutput(this); 

            if (itemstack == null) return false;
            if (this.contents[1] == null) return true;

            if (!this.contents[1].isItemEqual(itemstack)) return false;

            int result = contents[1].stackSize + itemstack.stackSize;
            return result <= getInventoryStackLimit() && result <= this.contents[1].getMaxStackSize();
        }
    }

    public void update() 
    {
        if (this.isBurning() && this.canSmelt()) {
            ++this.workingTime;
        }
        if (this.canSmelt() && !this.isBurning()) {
            this.workingTime = 1;

        }
        if (this.canSmelt() && this.workingTime == this.workingTimeNeeded) {
            this.smeltItem();
            this.workingTime = 0;
        }
        if (!this.canSmelt()) {
            this.workingTime = 0;
        }
    }

    public void smeltItem() {
        if (this.canSmelt()) {
            ItemStack itemstack = AnalyzerRecipes.getOutput(this); 
            if (this.contents[1] == null) 
            {
                this.contents[1] = itemstack.copy();
            } 
            else if (this.contents[1].getItem() == itemstack.getItem()) 
            {
                this.contents[1].stackSize += itemstack.stackSize;
            }

            --this.contents[0].stackSize;
           

            if (this.contents[0].stackSize <= 0) 
            {
                this.contents[0] = null;
            }
           

        }
    }

    @SideOnly(Side.CLIENT)
    public int getCookProgress() {
        return this.workingTime * 40 / this.workingTimeNeeded;
    }
    
	@Override
	public String getName() 
	{
		return "tile.BlockAnalyzer";
	}

	@Override
	public boolean hasCustomName() 
	{
		return false;
	}

	@Override
	public IChatComponent getDisplayName()
	{
		return null;
	}

	@Override
	public int getInventoryStackLimit() 
	{
		return 64;
	}

	@Override
	public boolean isUseableByPlayer(EntityPlayer player) 
	{
       return this.worldObj.getTileEntity(this.pos) != this ? false : player.getDistanceSq((double)this.pos.getX() + 0.5D, (double)this.pos.getY() + 0.5D, (double)this.pos.getZ() + 0.5D) <= 64.0D;
	}

	@Override
	public void openInventory(EntityPlayer player) {}

	@Override
	public void closeInventory(EntityPlayer player) {}

	@Override
	public int getField(int id) 
	{
		 switch (id)
	        {
	            case 0:
	                return this.workingTime;
	            case 1:
	                return this.workingTimeNeeded;
	            default:
	                return 0;
	        }
	}

	@Override
	public void setField(int id, int value) 
	{
		 switch (id)
	        {
	            case 0:
	                this.workingTime = value;
	                break;
	            case 1:
	                this.workingTimeNeeded = value;
	        }
	}

	@Override
	public int getFieldCount() 
	{
		return 2;
	}

	@Override
	public void clear() 
	{
		 for (int i = 0; i < this.contents.length; ++i)
	        {
	            this.contents[i] = null;
	        }
	}


}
 
