package net.spyman.utils.common.inventory;

import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.inventory.IInventory;
import net.minecraft.item.ItemStack;
import net.minecraft.item.crafting.FurnaceRecipes;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.nbt.NBTTagList;
import net.minecraft.util.text.ITextComponent;
import net.minecraftforge.fml.relauncher.Side;
import net.minecraftforge.fml.relauncher.SideOnly;
import net.spyman.utils.common.items.ItemPortableFurnace;

import javax.annotation.Nullable;

public class InventoryPortableFurnace implements IInventory
{
    private ItemStack[] inventorySlots = new ItemStack[3];
    public int workingTime = 0;
    public int workingTimeNeeded = 200;
    public int workingTimeNeededDefault = 200;
    public int burnTime = 0;
    public int burnTimeTotal = 0;
    public int inventorySize = inventorySlots.length;
    public int fuelSlot = 1;
    public int outputSlot = 2;
    public int inputSlot = 0;

    public InventoryPortableFurnace(ItemStack itemStack)
    {
        if (!itemStack.hasTagCompound())
        {
            itemStack.setTagCompound(new NBTTagCompound());
        }

        this.readFromNBT(itemStack.getTagCompound());
    }

    public void readFromNBT(NBTTagCompound compound)
    {
        NBTTagList nbttaglist = compound.getTagList("MechanicalCraftItems", 10);
        this.inventorySlots = new ItemStack[this.getSizeInventory()];

        for(int i = 0; i < nbttaglist.tagCount(); ++i)
        {
            NBTTagCompound nbtTagCompound1 = nbttaglist.getCompoundTagAt(i);
            int j = nbtTagCompound1.getByte("Slot") & 255;

            if(j >= 0 && j < this.inventorySlots.length)
            {
                this.inventorySlots[j] = ItemStack.loadItemStackFromNBT(nbtTagCompound1);
            }
        }
        this.workingTime = compound.getShort("workingTime");
        this.burnTime = compound.getShort("burnTime");
        this.burnTimeTotal = compound.getShort("burnTimeTotal");
    }

    public void writeToNBT(NBTTagCompound compound)
    {
        NBTTagList nbttaglist = new NBTTagList();
        for(int i = 0; i < this.inventorySlots.length; ++i)
        {
            if(this.inventorySlots[i] != null)
            {
                NBTTagCompound nbtTagCompound1 = new NBTTagCompound();
                nbtTagCompound1.setByte("Slot", (byte)i);
                this.inventorySlots[i].writeToNBT(nbtTagCompound1);
                nbttaglist.appendTag(nbtTagCompound1);

            }
        }
        compound.setTag("MechanicalCraftItems", nbttaglist);
        compound.setShort("workingTime", (short)this.workingTime);
        compound.setShort("burnTime", (short)this.burnTime);
        compound.setShort("burnTimeTotal", (short)this.burnTimeTotal);
    }

    @Override
    public int getSizeInventory()
    {
        return this.inventorySize;
    }

    @Nullable
    @Override
    public ItemStack getStackInSlot(int index)
    {
        return this.inventorySlots[index];
    }

    @Nullable
    @Override
    public ItemStack decrStackSize(int index, int amount)
    {
        if(this.inventorySlots[index] != null)
        {
            ItemStack itemstack;

            if(this.inventorySlots[index].stackSize <= amount)
            {
                itemstack = this.inventorySlots[index];
                this.inventorySlots[index] = null;
                this.markDirty();
                return itemstack;
            }
            else
            {
                itemstack = this.inventorySlots[index].splitStack(amount);

                if(this.inventorySlots[index].stackSize == 0)
                {
                    this.inventorySlots[index] = null;
                }
                this.markDirty();
                return itemstack;
            }
        }
        else
        {
            return null;
        }
    }

    @Nullable
    @Override
    public ItemStack removeStackFromSlot(int index)
    {
        return null;
    }

    @Override
    public void setInventorySlotContents(int index, @Nullable ItemStack stack)
    {
        this.inventorySlots[index] = stack;
    }

    @Override
    public int getInventoryStackLimit()
    {
        return 64;
    }

    @Override
    public void markDirty()
    {

    }

    @Override
    public boolean isUseableByPlayer(EntityPlayer player)
    {
        return true;
    }

    @Override
    public void openInventory(EntityPlayer player)
    {

    }

    @Override
    public void closeInventory(EntityPlayer player)
    {

    }

    @Override
    public boolean isItemValidForSlot(int index, ItemStack stack)
    {
        return !(stack.getItem() instanceof ItemPortableFurnace);
    }

    public boolean isBurning()
    {
        return this.workingTime > 0;
    }

    protected boolean canSmelt()
    {
        if(this.inventorySlots[this.inputSlot] == null)
        {
            return false;
        }
        else
        {
            ItemStack itemstack = FurnaceRecipes.instance().getSmeltingResult(inventorySlots[this.inputSlot]);

            if(itemstack == null)
            {
                return false;
            }
            if(this.inventorySlots[this.outputSlot] == null)
            {
                return true;
            }
            if(!this.inventorySlots[this.outputSlot].isItemEqual(itemstack))
            {
                return false;
            }

            int result = inventorySlots[this.outputSlot].stackSize + itemstack.stackSize;
            return result <= getInventoryStackLimit() && result <= this.inventorySlots[2].getMaxStackSize();
        }
    }

    public void smeltItem()
    {
        if (this.canSmelt())
        {
            ItemStack itemstack = FurnaceRecipes.instance().getSmeltingResult(inventorySlots[this.inputSlot]);
            if (this.inventorySlots[this.outputSlot] == null)
            {
                this.inventorySlots[this.outputSlot] = itemstack.copy();
            }
            else if (this.inventorySlots[this.outputSlot].getItem() == itemstack.getItem())
            {
                this.inventorySlots[this.outputSlot].stackSize += itemstack.stackSize;
            }
            this.decrStackSize(0, 1);
        }
    }

    @Override
    public int getField(int id)
    {
        return 0;
    }

    @Override
    public void setField(int id, int value)
    {

    }

    @Override
    public int getFieldCount()
    {
        return 0;
    }

    @Override
    public void clear()
    {

    }

    @Override
    public String getName()
    {
        return "item.container.portable_furnace.gui";
    }

    @Override
    public boolean hasCustomName()
    {
        return false;
    }

    @Override
    public ITextComponent getDisplayName()
    {
        return null;
    }

    @SideOnly(Side.CLIENT)
    public int getSmeltProcess()
    {
        return this.workingTime * 24 / this.workingTimeNeeded;
    }

    @SideOnly(Side.CLIENT)
    public int getBurnTime()
    {
        return this.burnTime * 14 / this.burnTimeTotal;
    }
}


