package net.spyman.basicmachines.common.blocks.tileentity;

import net.minecraft.block.material.MaterialLiquid;
import net.minecraft.block.state.IBlockState;
import net.minecraft.init.Blocks;
import net.minecraft.inventory.IInventory;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.EnumParticleTypes;
import net.minecraft.util.ITickable;
import net.minecraft.util.SoundCategory;
import net.minecraft.util.math.BlockPos;
import net.spyman.basicmachines.common.blocks.BMBlocks;

import java.util.ArrayList;
import java.util.List;

public class TEBlockQuarry extends TileEntity implements ITickable
{
    private boolean isActive = false;
    private List<BlockPos> blockList = new ArrayList<BlockPos>();
    private IInventory storage;
    private int blockListIndex = 0;
    private int timer = 0;
    private int miningSpeed = 10;

    private int stopY = 0;
    private int maxY = 255;
    private int sizeX = 16;
    private int sizeZ = 16;
    private int maxSizeX = 64;
    private int maxSizeZ = 64;

    @Override
    public void readFromNBT(NBTTagCompound c)
    {
        this.setActivated(c.getBoolean("activated"));
    }

    @Override
    public NBTTagCompound writeToNBT(NBTTagCompound c)
    {
        c.setBoolean("activated", this.isActive);
        return c;
    }

    /**
     * Define area bounds to mine
     *
     * @param sx    Area X size
     * @param sz    Area Z size
     * @param stopY Area Y (Quarry stop to work at this layer)
     */
    public void setAreaBounds(int sx, int sz, int stopY)
    {
        if (sx < this.maxSizeX)
            this.sizeX = sx;
        else this.sizeX = this.maxSizeX;

        if (sz < this.maxSizeZ)
            this.sizeZ = sz;
        else this.sizeZ = this.maxSizeZ;

        if (stopY > 0 && stopY < this.maxY)
            this.stopY = stopY;
        else this.stopY = 0;
    }

    public int getStopY()
    {
        return this.stopY;
    }

    public void setStopY(int stopY)
    {
        this.stopY = stopY;
    }

    public int getSizeX()
    {
        return this.sizeX;
    }

    public void setSizeX(int sizeX)
    {
        this.sizeX = sizeX;
    }

    public int getSizeZ()
    {
        return this.sizeZ;
    }

    public void setSizeZ(int sizeZ)
    {
        this.sizeZ = sizeZ;
    }

    public boolean isActivated()
    {
        return this.isActive;
    }

    public void setActivated(boolean activated)
    {
        this.isActive = activated;

        if (!activated)
        {
            this.blockListIndex = 0;
            this.getBlockList().clear();
        }
        else
            this.prepare();
    }

    @Override
    public void update()
    {
        this.timer++;

        if (this.timer == this.miningSpeed)
        {
            this.timer = 0;

            if (this.isActive)
            {
                if (this.hasIInventory())
                {
                    if (this.blockListIndex < this.blockList.size())
                        this.mine(this.blockListIndex);
                    else this.setActivated(false);
                }
            }
        }
    }

    /**
     * Put all block's BlockPos in range in a list ("blockList").
     */
    private void prepare()
    {
        for (int y = this.pos.getY() - 1; y > 0; y--)
        {
            for (int x = this.pos.getX() - 8; x < (this.pos.getX() + 8); x++)
            {
                for (int z = this.pos.getZ() - 8; z < (this.pos.getZ() + 8); z++)
                {
                    BlockPos bp = new BlockPos(x, y, z);
                    IBlockState b = this.world.getBlockState(bp);

                    if (this.isValidBlock(b, bp))
                        this.blockList.add(bp);
                }
            }
        }
    }

    public List<BlockPos> getBlockList()
    {
        return this.blockList;
    }

    /**
     * Return true if and IInventory is available on the top of Quarry, return false if that does'nt exist
     *
     * @return
     */
    public boolean hasIInventory()
    {
        TileEntity te = this.world.getTileEntity(new BlockPos(this.pos.getX(), this.pos.getY() + 1, this.pos.getZ()));

        if (te != null && te instanceof IInventory)
        {
            this.storage = (IInventory) te;
            return true;
        }
        else
        {
            this.storage = null;
            return false;
        }
    }

    /**
     * Call this to mine the next block in "blockList"
     *
     * @param index
     */
    private void mine(int index)
    {
        BlockPos pos1 = this.blockList.get(index);
        IBlockState b = this.world.getBlockState(pos1);

        if (this.isValidBlock(b, pos1))
        {
            if (this.storeItemInInventory(new ItemStack(b.getBlock(), 1, b.getBlock().getMetaFromState(b))))
            {
                this.playEffect(index, b);
                this.world.setBlockToAir(pos1);
                this.blockListIndex++;
            }
            else
                this.setActivated(false);
        }
        else
            this.blockListIndex++;
    }

    private boolean isValidBlock(IBlockState b, BlockPos pos)
    {
        return (b.getBlock() != null && b.getBlock() != Blocks.BEDROCK && b.getBlock() != Blocks.AIR && b.getBlock() != BMBlocks.BLOCK_QUARRY && !(this.world.getTileEntity(pos) instanceof IInventory) && !(b.getBlock().getMaterial(b) instanceof MaterialLiquid));
    }

    /**
     * Return true if an item can be stored in a IInventory on the Quarry, return false if that impossible.
     *
     * @param s itemstack to store
     * @return
     */
    private boolean storeItemInInventory(ItemStack s)
    {
        if (this.hasIInventory())
        {
            for (int i = 0; i < this.storage.getSizeInventory(); i++)
            {
                ItemStack s1 = this.storage.getStackInSlot(i);

                if (s1.getItem() == s.getItem() && s1.getCount() < s.getMaxStackSize() && s1.getMetadata() == s.getMetadata())
                {
                    s1.grow(1);
                    return true;
                }
                else if (s1.isEmpty())
                {
                    this.storage.setInventorySlotContents(i, s);
                    return true;
                }
            }
        }

        return false;
    }

    private void playEffect(int index, IBlockState b)
    {
        this.world.playSound(null, this.blockList.get(index), b.getBlock().getSoundType().getBreakSound(), SoundCategory.BLOCKS, b.getBlock().getSoundType().volume, b.getBlock().getSoundType().pitch);
        double px = this.blockList.get(index).getX();
        double py = this.blockList.get(index).getY() + 0.25D;
        double pz = this.blockList.get(index).getZ();
        this.world.spawnParticle(EnumParticleTypes.SMOKE_LARGE, px, py, pz, 0.0D, 0.0D, 0.0D);
    }
}