package com.CSC.net.block;

import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.init.Items;
import net.minecraft.inventory.ISidedInventory;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.nbt.NBTTagList;
import net.minecraft.network.NetworkManager;
import net.minecraft.network.Packet;
import net.minecraft.network.play.server.S35PacketUpdateTileEntity;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.ChatComponentText;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.IChatComponent;
import net.minecraft.util.ITickable;
import net.minecraftforge.fml.relauncher.Side;
import net.minecraftforge.fml.relauncher.SideOnly;

public class TileEntityMachineLaver extends TileEntity implements ITickable, ISidedInventory
{
    private String customName;
    public static final int INPUTS_SLOTS = 4;
    public static final int FUEL_SLOTS = 1;
    public static final int OUTPUT_SLOTS = 4;
    private ItemStack[] contents = new ItemStack[INPUTS_SLOTS + FUEL_SLOTS + OUTPUT_SLOTS];
    private int currentWorkingTime = 0;
    private int totalWorkingTime = 300;
    private int remainingFuelTime = 0;
    private int totalCurrentFuelTime;

    @Override
    public void readFromNBT(NBTTagCompound compound)
    {
        super.readFromNBT(compound);
        NBTTagList nbttaglist = compound.getTagList("Items", 10);
        this.contents = new ItemStack[this.getSizeInventory()];

        for(int i = 0; i < nbttaglist.tagCount(); ++i)
        {
            NBTTagCompound nbttagcompound1 = nbttaglist.getCompoundTagAt(i);
            byte b0 = nbttagcompound1.getByte("Slot");

            if(b0 >= 0 && b0 < this.contents.length)
            {
                this.contents[b0] = ItemStack.loadItemStackFromNBT(nbttagcompound1);
            }
        }

        this.currentWorkingTime = compound.getShort("CurrentWorkingTime");
        this.totalWorkingTime = compound.getShort("TotalWorkingTime");
        this.remainingFuelTime = compound.getShort("RemainingFuelTime");
        this.totalCurrentFuelTime = compound.getShort("TotalCurrentFuelTime");

        if(compound.hasKey("CustomName", 8))
        {
            this.customName = compound.getString("CustomName");
        }
    }

    @Override
    public void writeToNBT(NBTTagCompound compound)
    {
        super.writeToNBT(compound);

        compound.setShort("CurrentWorkingTime", (short)this.currentWorkingTime);
        compound.setShort("TotalWorkingTime", (short)this.totalWorkingTime);
        compound.setShort("RemainingFuelTime", (short)this.remainingFuelTime);
        compound.setShort("TotalCurrentFuelTime", (short)this.totalCurrentFuelTime);

        NBTTagList nbttaglist = new NBTTagList();

        for(int i = 0; i < this.contents.length; ++i)
        {
            if(this.contents[i] != null)
            {
                NBTTagCompound nbttagcompound1 = new NBTTagCompound();
                nbttagcompound1.setByte("Slot", (byte)i);
                this.contents[i].writeToNBT(nbttagcompound1);
                nbttaglist.appendTag(nbttagcompound1);
            }
        }

        compound.setTag("Items", nbttaglist);

        if(this.hasCustomName())
        {
            compound.setString("CustomName", this.customName);
        }
    }

    @Override
    public int getSizeInventory()
    {
        return this.contents.length;
    }

    @Override
    public ItemStack getStackInSlot(int index)
    {
        return this.contents[index];
    }

    @Override
    public ItemStack decrStackSize(int index, int count)
    {
        if(this.contents[index] != null)
        {
            ItemStack itemstack;

            if(this.contents[index].stackSize <= count)
            {
                itemstack = this.contents[index];
                this.contents[index] = null;
                this.markDirty();
                return itemstack;
            }
            else
            {
                itemstack = this.contents[index].splitStack(count);

                if(this.contents[index].stackSize == 0)
                {
                    this.contents[index] = null;
                }
                this.markDirty();
                return itemstack;
            }
        }
        else
        {
            return null;
        }
    }

    @Override
    public ItemStack removeStackFromSlot(int index)
    {
        if(this.contents[index] != null)
        {
            ItemStack itemstack = this.contents[index];
            this.contents[index] = null;
            return itemstack;
        }
        else
        {
            return null;
        }
    }

    @Override
    public void setInventorySlotContents(int index, ItemStack stack)
    {
        this.contents[index] = stack;
        if(stack != null && stack.stackSize > this.getInventoryStackLimit())
        {
            stack.stackSize = this.getInventoryStackLimit();
        }
    }

    @Override
    public int getInventoryStackLimit()
    {
        return 64;
    }

    @Override
    public boolean isUseableByPlayer(EntityPlayer player)
    {
        return this.worldObj.getTileEntity(this.pos) != this ? false : player.getDistanceSq((double)this.pos.getX() + 0.5D, (double)this.pos.getY() + 0.5D, (double)this.pos.getZ() + 0.5D) <= 64.0D;
    }

    @Override
    public void openInventory(EntityPlayer player)
    {}

    @Override
    public void closeInventory(EntityPlayer player)
    {}

    @Override
    public boolean isItemValidForSlot(int index, ItemStack stack)
    {
        return index >= INPUTS_SLOTS + FUEL_SLOTS ? false : index >= INPUTS_SLOTS && !this.isFuel(stack) ? false : true;
    }

    @Override
    public int getField(int id)
    {
        switch(id)
        {
            case 0:
                return this.currentWorkingTime;
            case 1:
                return this.totalWorkingTime;
            case 2:
                return this.remainingFuelTime;
            case 3:
                return this.totalCurrentFuelTime;
            default:
                return 0;
        }
    }

    @Override
    public void setField(int id, int value)
    {
        switch(id)
        {
            case 0:
                this.currentWorkingTime = value;
                break;
            case 1:
                this.totalWorkingTime = value;
                break;
            case 2:
                this.remainingFuelTime = value;
                break;
            case 3:
                this.totalCurrentFuelTime = value;
        }
    }

    @Override
    public int getFieldCount()
    {
        return 3;
    }

    @Override
    public void clear()
    {
        for(int i = 0; i < this.contents.length; i++)
        {
            this.contents[i] = null;
        }
    }

    @Override
    public String getName()
    {
        return this.hasCustomName() ? this.customName : "Machine � laver";
    }

    @Override
    public boolean hasCustomName()
    {
        return this.customName != null && this.customName.length() > 0;
    }

    public void setCustomInventoryName(String name)
    {
        this.customName = name;
    }

    @Override
    public IChatComponent getDisplayName()
    {
        return new ChatComponentText(this.getName());
    }

    @Override
    public int[] getSlotsForFace(EnumFacing side)
    {
        if(side == EnumFacing.UP)
            return this.getInputSlotsIds();
        if(side == EnumFacing.DOWN)
            return this.getOutputSlotsIds();
        return this.getFuelSlotsIds();
    }

    @Override
    public boolean canInsertItem(int index, ItemStack stack, EnumFacing direction)
    {
        return this.isItemValidForSlot(index, stack);
    }

    @Override
    public boolean canExtractItem(int index, ItemStack stack, EnumFacing direction)
    {
        for(int id : this.getOutputSlotsIds())
        {
            if(id == index)
                return true;
        }
        return false;
    }

    @Override
    public void update()
    {
        if(this.isRemainingFuel())
        {
            --this.remainingFuelTime;
        }
        if(this.isWorking() && this.canWork())
        {
            ++this.currentWorkingTime;
            blockmachinelaver.isBurning = true;
        }
        if(!this.isWorking() && this.canWork())
        {
            this.currentWorkingTime = 1;
            blockmachinelaver.isBurning = true;
        }
        if(this.canWork() && this.currentWorkingTime >= this.totalWorkingTime)
        {
            this.applyRecipe();
            this.currentWorkingTime = 0;
        }
        if(!this.canWork())
        {
            this.currentWorkingTime = 0;
            blockmachinelaver.isBurning = false;
        }
        this.markDirty();
    }

    private boolean isFuel(ItemStack stack)
    {
        return stack == null ? false : stack.getItem() == Items.blaze_powder;
    }

    private int[] getInputSlotsIds()
    {
        int[] ids = new int[INPUTS_SLOTS];
        for(int i = 0; i < INPUTS_SLOTS; i++)
        {
            ids[i] = i;
        }
        return ids;
    }

    private int[] getFuelSlotsIds()
    {
        int[] ids = new int[FUEL_SLOTS];
        int k = 0;
        for(int i = INPUTS_SLOTS; i < (INPUTS_SLOTS + FUEL_SLOTS); i++)
        {
            ids[k] = i;
            k++;
        }
        return ids;
    }

    private int[] getOutputSlotsIds()
    {
        int[] ids = new int[OUTPUT_SLOTS];
        int k = 0;
        for(int i = (INPUTS_SLOTS + FUEL_SLOTS); i < (INPUTS_SLOTS + FUEL_SLOTS + OUTPUT_SLOTS); i++)
        {
            ids[k] = i;
            k++;
        }
        return ids;
    }

    private ItemStack[] getInputSlotsContents()
    {
        ItemStack[] s = new ItemStack[INPUTS_SLOTS];
        for(int i = 0; i < INPUTS_SLOTS; i++)
        {
            s[i] = this.getStackInSlot(i);
        }
        return s;
    }

    private ItemStack[] getOutputSlotsContents()
    {
        ItemStack[] s = new ItemStack[OUTPUT_SLOTS];
        int k = 0;
        for(int i = (INPUTS_SLOTS + FUEL_SLOTS); i < (INPUTS_SLOTS + FUEL_SLOTS + OUTPUT_SLOTS); i++)
        {
            s[k] = this.getStackInSlot(i);
            k++;
        }
        return s;
    }

    public boolean isWorking()
    {
        return this.currentWorkingTime > 0;
    }

    private boolean hasNullInput()
    {
        for(int i = 0; i < INPUTS_SLOTS; i++)
        {
            if(this.getStackInSlot(i) == null)
                return true;
        }
        return false;
    }

    private boolean areOutputsNull()
    {
        for(int i = (INPUTS_SLOTS + FUEL_SLOTS); i < (INPUTS_SLOTS + FUEL_SLOTS + OUTPUT_SLOTS); i++)
        {
            if(this.getStackInSlot(i) != null)
                return false;
        }
        return true;
    }

    private boolean canCombineStacks(ItemStack[] stack1, ItemStack[] stack2)
    {
        if(stack1.length != stack2.length)
            return false;
        for(int i = 0; i < stack1.length; i++)
        {
            if(!this.canCombineItems(stack1[i], stack2[i]))
            {
                return false;
            }
        }
        return true;
    }

    private boolean canCombineItems(ItemStack item1, ItemStack item2)
    {
        if(item1 == null)
            return true;
        if(!item1.isItemEqual(item2))
            return false;
        int result = item1.stackSize + item2.stackSize;
        return result <= getInventoryStackLimit() && result <= item1.getMaxStackSize();
    }

    private boolean outOfFuel()
    {
        if(!this.isRemainingFuel())
        {
            for(int i = INPUTS_SLOTS; i < (INPUTS_SLOTS + FUEL_SLOTS); i++)
            {
                if(this.getStackInSlot(i) != null)
                {
                    int duration = RecipesMachineLaver.instance().getFuelDuration(this.getStackInSlot(i));
                    if(duration > 0)
                    {
                        this.remainingFuelTime += duration;
                        this.totalCurrentFuelTime = duration;
                        this.decrStackSize(i, 1);
                    }
                }
            }
        }
        this.markDirty();
        return !this.isRemainingFuel();
    }

    public boolean isRemainingFuel()
    {
        return this.remainingFuelTime > 0;
    }

    private boolean canWork()
    {
        if(this.hasNullInput())
        {
            return false;
        }
        ItemStack[] itemstacks = RecipesMachineLaver.instance().getRecipeResult(this.getInputSlotsContents());
        if(itemstacks == null)
        {
            return false;
        }
        if(this.outOfFuel())
            return false;
        if(this.areOutputsNull())
            return true;
        if(!this.canCombineStacks(this.getOutputSlotsContents(), itemstacks))
            return false;
        return true;
    }

    private void applyRecipe()
    {
        ItemStack[] itemstacks = RecipesMachineLaver.instance().getRecipeResult(this.getInputSlotsContents());
        if(this.areOutputsNull())
        {
            int k = 0;
            for(int i = (INPUTS_SLOTS + FUEL_SLOTS); i < (INPUTS_SLOTS + FUEL_SLOTS + OUTPUT_SLOTS); i++)
            {
                this.setInventorySlotContents(i, itemstacks[k].copy());
                k++;
            }
        }
        else
        {
            int k = 0;
            for(int i = (INPUTS_SLOTS + FUEL_SLOTS); i < (INPUTS_SLOTS + FUEL_SLOTS + OUTPUT_SLOTS); i++)
            {
                if(this.getStackInSlot(i) != null && itemstacks[k] != null)
                {
                    this.getStackInSlot(i).stackSize += itemstacks[k].copy().stackSize;
                }
                else if(this.getStackInSlot(i) == null)
                {
                    this.setInventorySlotContents(i, itemstacks[k].copy());;
                }
                k++;
            }
        }
        for(int i = 0; i < INPUTS_SLOTS; i++)
        {
            this.decrStackSize(i, 1);
        }
        this.markDirty();
    }

    @Override
    public Packet getDescriptionPacket()
    {
        NBTTagCompound nbttagcompound = new NBTTagCompound();
        this.writeToNBT(nbttagcompound);
        return new S35PacketUpdateTileEntity(this.getPos(), this.getBlockMetadata(), nbttagcompound);
    }

    @Override
    public void onDataPacket(NetworkManager net, S35PacketUpdateTileEntity pkt)
    {
        this.readFromNBT(pkt.getNbtCompound());
        this.worldObj.markBlockRangeForRenderUpdate(this.pos, this.pos);
    }

    @SideOnly(Side.CLIENT)
    public int getRecipeProgress()
    {
        return this.currentWorkingTime * 17 / this.totalWorkingTime;
    }

    @SideOnly(Side.CLIENT)
    public int getFuelProgress()
    {
        return this.remainingFuelTime * 20 / this.totalCurrentFuelTime;
    }

}
